/**************************************************************************************

Copyright (c) Hilscher Gesellschaft fuer Systemautomation mbH. All Rights Reserved.

***************************************************************************************

  $Id:  $:

  Description:
    Implementation of the System device class.
    Derived from CCifXDeviceBase to handle a system channel connection.

  Changes:
    Date        Description
    -----------------------------------------------------------------------------------

    2008-11-25  GetSystemStatusBlock()
                - LEDs removed
                - CPU load and SystemStartTime values added
    2008-02-08  Communication Channel information extended
                by Communication/Protocol/Conformance class
    2006-06-28  initial version

**************************************************************************************/

#include "StdAfx.h"
#include ".\cifxdevicesys.h"
#include "CifxErrors.h"
#include "Hil_DualPortMemory.h"

///////////////////////////////////////////////////////////////////////////////////////////
/// \file CifXDeviceSys.cpp
///  Implementation of the System device class
///////////////////////////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////////
/// Constructuor
///   \param ptBoardInfo    Pointer to board information from cifX device driver
///   \param ptChannelInfo  Pointer to channel information from cifX device driver
/////////////////////////////////////////////////////////////////////////////
CCifXDeviceSys::CCifXDeviceSys(BOARD_INFORMATION* ptBoardInfo, CHANNEL_INFORMATION* ptChannelInfo, CCifXAPIlib* pcCifXAPI)
: CCifXDeviceBase(ptBoardInfo, ptChannelInfo, pcCifXAPI)
, m_hSysChannel(NULL)
{
}

/////////////////////////////////////////////////////////////////////////////
/// Destructor
/////////////////////////////////////////////////////////////////////////////
CCifXDeviceSys::~CCifXDeviceSys(void)
{
  CloseDevice();
}

/////////////////////////////////////////////////////////////////////////////
/// Overridden query function to identify a system device (which does not allow I/O, etc)
///   \return true
/////////////////////////////////////////////////////////////////////////////
bool CCifXDeviceSys::IsSystemDevice(void)
{
  return true;
}

/////////////////////////////////////////////////////////////////////////////
/// Overridden Open Device function. Opens a connection to the device via cifX Driver
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CCifXDeviceSys::OpenDevice(void)
{
  if(NULL != m_hSysChannel)
    CloseDevice();

  return m_pcCifXAPI->SysdeviceOpen(m_tBoardInfo.abBoardName, &m_hSysChannel);
}

/////////////////////////////////////////////////////////////////////////////
/// Overridden Close Device function. Closes a connection to the device via cifX Driver
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CCifXDeviceSys::CloseDevice(void)
{
  int32_t lRet = CIFX_NO_ERROR;

  if(NULL != m_hSysChannel)
  {
    lRet = m_pcCifXAPI->xSysdeviceClose(m_hSysChannel);
    m_hSysChannel = NULL;
  }

  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Overridden Put Packet function. Puts a packet to the device mailbox
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CCifXDeviceSys::PutPacket(CIFX_PACKET* ptPacket, uint32_t ulTimeout)
{
  return m_pcCifXAPI->xSysdevicePutPacket(m_hSysChannel, ptPacket, ulTimeout);
}

/////////////////////////////////////////////////////////////////////////////
/// Overridden Get Packet function. Gets a packet to the device mailbox
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CCifXDeviceSys::GetPacket(CIFX_PACKET* ptPacket, uint32_t ulBufferSize, uint32_t ulTimeout)
{
  return m_pcCifXAPI->xSysdeviceGetPacket(m_hSysChannel,ulBufferSize, ptPacket, ulTimeout);
}

/////////////////////////////////////////////////////////////////////////////
/// Retrieves the System Info Block on a system device
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CCifXDeviceSys::GetSystemInfoBlock(PROPERTY_VECTOR& cvRet)
{
  SYSTEM_CHANNEL_SYSTEM_INFO_BLOCK tSysInfo = {0};

  int32_t lRet = m_pcCifXAPI->xSysdeviceInfo(m_hSysChannel,
                                          CIFX_INFO_CMD_SYSTEM_INFO_BLOCK, 
                                          sizeof(tSysInfo), 
                                          &tSysInfo);

  if(CIFX_NO_ERROR == lRet)
  {
    PROPERTY_LIST tList;

    tList.csProperty = _T("Cookie");
    tList.csValue.Empty();
    for(unsigned char bIdx = 0; bIdx < sizeof(tSysInfo.abCookie) / sizeof(tSysInfo.abCookie[0]); ++bIdx)
      tList.csValue.AppendFormat(_T("%c"), (char)tSysInfo.abCookie[bIdx]);
    cvRet.push_back(tList);
/*
    tList.csProperty = _T("System Error");
    tList.csValue.Format(_T("0x%08X"), tSysInfo.ulSystemError);
    cvRet.push_back(tList);
*/    
    tList.csProperty = _T("DPM Total Size");
    tList.csValue.Format(_T("%u"), tSysInfo.ulDpmTotalSize);
    cvRet.push_back(tList);

/*
    tList.csProperty = _T("Mailbox Size");
    tList.csValue.Format(_T("%u"), tSysInfo.ulMBXSize);
    cvRet.push_back(tList);
*/
    tList.csProperty = _T("Device Number");
    tList.csValue.Format(_T("%u"), tSysInfo.ulDeviceNumber);
    cvRet.push_back(tList);

    tList.csProperty = _T("Serial Number");
    tList.csValue.Format(_T("%u"), tSysInfo.ulSerialNumber);
    cvRet.push_back(tList);

    for(unsigned char bIdx = 0; bIdx < sizeof(tSysInfo.ausHwOptions) / sizeof(tSysInfo.ausHwOptions[0]); ++bIdx)
    {
      tList.csProperty.Format(_T("HW Option[%u]"), bIdx);
      tList.csValue.Format(_T("0x%04X"), tSysInfo.ausHwOptions[bIdx]);
      cvRet.push_back(tList);
    }

    tList.csProperty = _T("Manufacturer");
    tList.csValue.Format(_T("0x%04X"), tSysInfo.usManufacturer);
    cvRet.push_back(tList);

    tList.csProperty = _T("Production Date");
    tList.csValue.Format(_T("0x%04X"), tSysInfo.usProductionDate);
    cvRet.push_back(tList);

    tList.csProperty = _T("License Flags 1");
    tList.csValue.Format(_T("0x%08X"), tSysInfo.ulLicenseFlags1);
    cvRet.push_back(tList);

    tList.csProperty = _T("License Flags 2");
    tList.csValue.Format(_T("0x%08X"), tSysInfo.ulLicenseFlags2);
    cvRet.push_back(tList);

    tList.csProperty = _T("netX License ID");
    tList.csValue.Format(_T("0x%04X"), tSysInfo.usNetxLicenseID);
    cvRet.push_back(tList);

    tList.csProperty = _T("netX License Flags");
    tList.csValue.Format(_T("0x%04X"), tSysInfo.usNetxLicenseFlags);
    cvRet.push_back(tList);

    tList.csProperty = _T("Device Class");
    tList.csValue.Format(_T("0x%04X"), tSysInfo.usDeviceClass);
    cvRet.push_back(tList);

    tList.csProperty = _T("HW Revision");
    tList.csValue.Format(_T("%u"), tSysInfo.bHwRevision);
    cvRet.push_back(tList);

    tList.csProperty = _T("HW Compatibility");
    tList.csValue.Format(_T("%u"), tSysInfo.bHwCompatibility);
    cvRet.push_back(tList);

    tList.csProperty = _T("Device ID");
    tList.csValue.Format(_T("%u"), tSysInfo.bDevIdNumber);
    cvRet.push_back(tList);
  }

  return lRet;
}


/////////////////////////////////////////////////////////////////////////////
/// Retrieves the System Information on a system device
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CCifXDeviceSys::GetSystemInformation(PROPERTY_VECTOR& cvRet)
{
  SYSTEM_CHANNEL_SYSTEM_INFORMATION tSysInfo = {0};

  int32_t lRet = m_pcCifXAPI->xSysdeviceInfo(m_hSysChannel,
                                          CIFX_INFO_CMD_SYSTEM_INFORMATION, 
                                          sizeof(tSysInfo), 
                                          &tSysInfo);

  if(CIFX_NO_ERROR == lRet)
  {
    PROPERTY_LIST tList;
 
    tList.csProperty = _T("System Error");
    tList.csValue.Format(_T("0x%08X"), tSysInfo.ulSystemError);
    cvRet.push_back(tList);

    tList.csProperty = _T("DPM Total Size");
    tList.csValue.Format(_T("%u"), tSysInfo.ulDpmTotalSize);
    cvRet.push_back(tList);

    tList.csProperty = _T("Mailbox Size");
    tList.csValue.Format(_T("%u"), tSysInfo.ulMBXSize);
    cvRet.push_back(tList);

    tList.csProperty = _T("Device Number");
    tList.csValue.Format(_T("%u"), tSysInfo.ulDeviceNumber);
    cvRet.push_back(tList);

    tList.csProperty = _T("Serial Number");
    tList.csValue.Format(_T("%u"), tSysInfo.ulSerialNumber);
    cvRet.push_back(tList);

    tList.csProperty = _T("Open Count");
    tList.csValue.Format(_T("%u"), tSysInfo.ulOpenCnt);
    cvRet.push_back(tList);
  }

  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Retrieves the Channel Block on a system device
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CCifXDeviceSys::GetSystemChannelBlock(PROPERTY_VECTOR& cvRet)
{
  SYSTEM_CHANNEL_CHANNEL_INFO_BLOCK tInfo = {0};

  int32_t lRet = m_pcCifXAPI->xSysdeviceInfo(m_hSysChannel,
                                          CIFX_INFO_CMD_SYSTEM_CHANNEL_BLOCK, 
                                          sizeof(tInfo), 
                                          &tInfo);

  if(CIFX_NO_ERROR == lRet)
  {
    for(int iIdx = 0; iIdx < CIFX_MAX_NUMBER_OF_CHANNEL_DEFINITION; ++iIdx)
    {
      HIL_DPM_CHANNEL_INFO_BLOCK_T* ptChannel = (HIL_DPM_CHANNEL_INFO_BLOCK_T*)&tInfo.abInfoBlock[iIdx][0];
      PROPERTY_LIST                 tList;

      tList.csProperty.Format(_T("Channel #%u"), iIdx);
      tList.csValue = _T("--------------------------------");
      cvRet.push_back(tList);

      switch(ptChannel->tHandshake.bChannelType)
      {
      case HIL_CHANNEL_TYPE_APPLICATION:
        tList.csProperty = _T("Type");
        tList.csValue.Format(_T("Application (0x%02X)"), ptChannel->tApp.bChannelType);
        cvRet.push_back(tList);

        tList.csProperty = _T("Channel ID");
        tList.csValue.Format(_T("%u"), ptChannel->tApp.bChannelId);
        cvRet.push_back(tList);

        tList.csProperty = _T("Number of Blocks");
        tList.csValue.Format(_T("%u"), ptChannel->tApp.bNumberOfBlocks);
        cvRet.push_back(tList);

        tList.csProperty = _T("Size of Channel");
        tList.csValue.Format(_T("%u"), ptChannel->tApp.ulSizeOfChannel);
        cvRet.push_back(tList);

        tList.csProperty = _T("Handshake Width");
        tList.csValue    = ( (ptChannel->tApp.bSizePositionOfHandshake & HIL_HANDSHAKE_SIZE_MASK) == HIL_HANDSHAKE_SIZE_8BIT)?
                           _T("8 Bit") : _T("16 Bit");
        cvRet.push_back(tList);

        tList.csProperty = _T("Handshake Pos");
        tList.csValue    = ( (ptChannel->tApp.bSizePositionOfHandshake & HIL_HANDSHAKE_POSITION_MASK) == HIL_HANDSHAKE_POSITION_BEGINNING)?
                           _T("Start of Channel") : _T("Handshake Block");
        cvRet.push_back(tList);
        break;

      case HIL_CHANNEL_TYPE_HANDSHAKE:
        tList.csProperty = _T("Type");
        tList.csValue.Format(_T("Handshake (0x%02X)"), ptChannel->tHandshake.bChannelType);
        cvRet.push_back(tList);

        tList.csProperty = _T("Size of Channel");
        tList.csValue.Format(_T("%u"), ptChannel->tHandshake.ulSizeOfChannel);
        cvRet.push_back(tList);
        break;

      case HIL_CHANNEL_TYPE_COMMUNICATION:
        tList.csProperty = _T("Type");
        tList.csValue.Format(_T("Communication (0x%02X)"), ptChannel->tCom.bChannelType);
        cvRet.push_back(tList);

        tList.csProperty = _T("Channel ID");
        tList.csValue.Format(_T("%u"), ptChannel->tCom.bChannelId);
        cvRet.push_back(tList);

        tList.csProperty = _T("Number of Blocks");
        tList.csValue.Format(_T("%u"), ptChannel->tCom.bNumberOfBlocks);
        cvRet.push_back(tList);

        tList.csProperty = _T("Size of Channel");
        tList.csValue.Format(_T("%u"), ptChannel->tCom.ulSizeOfChannel);
        cvRet.push_back(tList);

        tList.csProperty = _T("Handshake Width");
        tList.csValue    = ( (ptChannel->tCom.bSizePositionOfHandshake & HIL_HANDSHAKE_SIZE_MASK) == HIL_HANDSHAKE_SIZE_8BIT)?
                           _T("8 Bit") : _T("16 Bit");
        cvRet.push_back(tList);

        tList.csProperty = _T("Handshake Pos");
        tList.csValue    = ( (ptChannel->tCom.bSizePositionOfHandshake & HIL_HANDSHAKE_POSITION_MASK) == HIL_HANDSHAKE_POSITION_BEGINNING)?
                           _T("Start of Channel") : _T("Handshake Block");
        cvRet.push_back(tList);
        
        tList.csProperty = _T("Communication Class");
        tList.csValue.Format(_T("0x%04X"), ptChannel->tCom.usCommunicationClass);
        cvRet.push_back(tList);

        tList.csProperty = _T("Protocol Class");
        tList.csValue.Format(_T("0x%04X"), ptChannel->tCom.usProtocolClass);
        cvRet.push_back(tList);
        
        tList.csProperty = _T("Protocol Conformance Class");
        tList.csValue.Format(_T("0x%04X"), ptChannel->tCom.usProtocolConformanceClass);
        cvRet.push_back(tList);
        break;

      case HIL_CHANNEL_TYPE_SYSTEM:
        tList.csProperty = _T("Type");
        tList.csValue.Format(_T("System (0x%02X)"), ptChannel->tSystem.bChannelType);
        cvRet.push_back(tList);

        tList.csProperty = _T("Number of Blocks");
        tList.csValue.Format(_T("%u"), ptChannel->tSystem.bNumberOfBlocks);
        cvRet.push_back(tList);

        tList.csProperty = _T("Size of Channel");
        tList.csValue.Format(_T("%u"), ptChannel->tSystem.ulSizeOfChannel);
        cvRet.push_back(tList);

        tList.csProperty = _T("Mailbox Start Offset");
        tList.csValue.Format(_T("%u"), ptChannel->tSystem.usMailboxStartOffset);
        cvRet.push_back(tList);

        tList.csProperty = _T("Mailbox Size");
        tList.csValue.Format(_T("%u"), ptChannel->tSystem.usSizeOfMailbox);
        cvRet.push_back(tList);

        tList.csProperty = _T("Handshake Width");
        tList.csValue    = ( (ptChannel->tSystem.bSizePositionOfHandshake & HIL_HANDSHAKE_SIZE_MASK) == HIL_HANDSHAKE_SIZE_8BIT)?
                           _T("8 Bit") : _T("16 Bit");
        cvRet.push_back(tList);

        tList.csProperty = _T("Handshake Pos");
        tList.csValue    = ( (ptChannel->tSystem.bSizePositionOfHandshake & HIL_HANDSHAKE_POSITION_MASK) == HIL_HANDSHAKE_POSITION_BEGINNING)?
                           _T("Start of Channel") : _T("Handshake Block");
        cvRet.push_back(tList);
        break;

      case HIL_CHANNEL_TYPE_UNDEFINED:
        tList.csProperty = _T("Type");
        tList.csValue.Format(_T("Undefined (0x%02X)"), ptChannel->tSystem.bChannelType);
        cvRet.push_back(tList);
        break;

      case HIL_CHANNEL_TYPE_NOT_AVAILABLE:
        tList.csProperty = _T("Type");
        tList.csValue.Format(_T("Not Available (0x%02X)"), ptChannel->tSystem.bChannelType);
        cvRet.push_back(tList);
        break;

      default:
        ASSERT(false);
        break;
      }
    }
  }

  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Retrieves the System Control Block on a system device
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CCifXDeviceSys::GetSystemControlBlock(PROPERTY_VECTOR& cvRet)
{
  SYSTEM_CHANNEL_SYSTEM_CONTROL_BLOCK tInfo = {0};

  int32_t lRet = m_pcCifXAPI->xSysdeviceInfo(m_hSysChannel,
                                          CIFX_INFO_CMD_SYSTEM_CONTROL_BLOCK, 
                                          sizeof(tInfo), 
                                          &tInfo);

  if(CIFX_NO_ERROR == lRet)
  {
    PROPERTY_LIST tList;

    tList.csProperty = _T("System Command COS");
    tList.csValue.Format(_T("0x%08X"), tInfo.ulSystemCommandCOS);
    cvRet.push_back(tList);

    tList.csProperty = _T("System Control");
    tList.csValue.Format(_T("0x%08X"), tInfo.ulSystemControl);
    cvRet.push_back(tList);
  }

  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Retrieves the System Control Block on a system device
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CCifXDeviceSys::GetSystemStatusBlock(PROPERTY_VECTOR& cvRet)
{
  SYSTEM_CHANNEL_SYSTEM_STATUS_BLOCK tInfo = {0};

  int32_t lRet = m_pcCifXAPI->xSysdeviceInfo(m_hSysChannel,
                                          CIFX_INFO_CMD_SYSTEM_STATUS_BLOCK, 
                                          sizeof(tInfo), 
                                          &tInfo);

  if(CIFX_NO_ERROR == lRet)
  {
    PROPERTY_LIST tList;

    tList.csProperty = _T("System COS");
    tList.csValue.Format(_T("0x%08X"), tInfo.ulSystemCOS);
    cvRet.push_back(tList);

    tList.csProperty = _T("System Status");
    tList.csValue.Format(_T("0x%08X"), tInfo.ulSystemStatus);
    cvRet.push_back(tList);

    tList.csProperty = _T("System Error");
    tList.csValue.Format(_T("0x%08X"), tInfo.ulSystemError);
    cvRet.push_back(tList);

    tList.csProperty = _T("Boot Error");
    tList.csValue.Format(_T("0x%08X"), tInfo.ulBootError);
    cvRet.push_back(tList);

    tList.csProperty = _T("System Start Time [s]");
    
    tList.csValue.Format(_T("%u (%ud %02u:%02u:%02u)"),  (tInfo.ulTimeSinceStart),
                                                         (tInfo.ulTimeSinceStart / 3600 / 24),
                                                         (tInfo.ulTimeSinceStart / 3600) % 24,
                                                         (tInfo.ulTimeSinceStart / 60) % 60,
                                                         (tInfo.ulTimeSinceStart % 60));
    cvRet.push_back(tList);
    
    tList.csProperty = _T("System CPU Load   [%]");
    tList.csValue.Format(_T("%u,%.2u"), tInfo.usCpuLoad / 100, tInfo.usCpuLoad % 100);
    cvRet.push_back(tList);
  }

  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Get Channel's Property/Value pairs for the Channel/Device
///   \return Property/Value pairs (strings)
/////////////////////////////////////////////////////////////////////////////
CCifXDeviceBase::PROPERTY_VECTOR CCifXDeviceSys::GetChannelInformation(uint32_t ulCmd)
{
  CCifXDeviceBase::PROPERTY_VECTOR cvRet;

  switch(ulCmd)
  {
  case CIFX_INFO_CMD_SYSTEM_INFO_BLOCK:
    GetSystemInfoBlock(cvRet);
    break;

  case CIFX_INFO_CMD_SYSTEM_INFORMATION:
    GetSystemInformation(cvRet);
    break;

  case CIFX_INFO_CMD_SYSTEM_CHANNEL_BLOCK:
    GetSystemChannelBlock(cvRet);
    break;

  case CIFX_INFO_CMD_SYSTEM_CONTROL_BLOCK:
    GetSystemControlBlock(cvRet);
    break;

  case CIFX_INFO_CMD_SYSTEM_STATUS_BLOCK:
    GetSystemStatusBlock(cvRet);
    break;

  }


  return cvRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Query the Name of the supplied channel information
///   \return Channel information description
/////////////////////////////////////////////////////////////////////////////
CString CCifXDeviceSys::GetChannelInfoName(void)
{
  return _T("System Channel Information");
}

/////////////////////////////////////////////////////////////////////////////
/// Query the Mailbox size of the device
///   \return Mailbox size in Bytes
/////////////////////////////////////////////////////////////////////////////
uint32_t CCifXDeviceSys::GetMailboxSize()
{
  SYSTEM_CHANNEL_SYSTEM_INFORMATION tSysInfo = {0};

  uint32_t ulRet = 0;

  if(CIFX_NO_ERROR == m_pcCifXAPI->xSysdeviceInfo(m_hSysChannel,
                                                  CIFX_INFO_CMD_SYSTEM_INFORMATION, 
                                                  sizeof(tSysInfo), 
                                                  &tSysInfo))
  {
    ulRet = tSysInfo.ulMBXSize;
  }

  return ulRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Download a file to the Channel/Device
///   \param ulChannel    Channel number
///   \param ulMode       Download Mode
///   \param szFileName   Short file name
///   \param pabFileData  File content
///   \param ulFileSize   Size of the file
///   \param pfnCallback  Progress Callback
///   \param pvUser       User parameter to pass to callback
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CCifXDeviceSys::Download(uint32_t ulChannel, uint32_t ulMode, char* szFileName, 
                              unsigned char* pabFileData, uint32_t ulFileSize, 
                              PFN_PROGRESS_CALLBACK pfnCallback, void* pvUser)
{
  return m_pcCifXAPI->xSysdeviceDownload(m_hSysChannel, ulChannel, ulMode, szFileName, pabFileData, ulFileSize, pfnCallback, NULL, pvUser);
}

/////////////////////////////////////////////////////////////////////////////
/// Get the current mailbox fill states
///   \param pulRecvCount Number of packet waiting to be received on device
///   \param pulSendCount Number of packets that can be send to the device
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CCifXDeviceSys::GetMBXState(uint32_t* pulRecvCount, uint32_t* pulSendCount)
{
  return m_pcCifXAPI->xSysdeviceGetMBXState(m_hSysChannel, pulRecvCount, pulSendCount);
}

/////////////////////////////////////////////////////////////////////////////
/// Performs a device reset (currently not defined)
///   \param ulMode     Reset mode
///   \param ulTimeout  Timeout to wait for reset to be performed
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CCifXDeviceSys::Reset(uint32_t ulMode, uint32_t ulTimeout, uint32_t ulResetParam)
{
  int32_t lRet = CIFX_NO_ERROR;

  if( CIFX_RESETEX_SYSTEMSTART == ulMode)
  {
    lRet = m_pcCifXAPI->xSysdeviceReset(m_hSysChannel, ulTimeout);
  } else
  {
    /* Construct reset mode for xSysdeviceResetEx */
    ulMode = ulMode | (ulResetParam << 4);
    lRet = m_pcCifXAPI->xSysdeviceResetEx(m_hSysChannel, ulTimeout, ulMode);
  }

  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Finds the first file entry
///   \param ulChannel        Channel number
///   \param ptDirectoryInfo  Pointer to directory info
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CCifXDeviceSys::FindFirstFile(uint32_t ulChannel, CIFX_DIRECTORYENTRY* ptDirectoryInfo)
{
  return m_pcCifXAPI->xSysdeviceFindFirstFile(m_hSysChannel, ulChannel, ptDirectoryInfo, NULL, NULL);
}

/////////////////////////////////////////////////////////////////////////////
/// Finds the next file entry
///   \param ulChannel        Channel number
///   \param ptDirectoryInfo  Pointer to directory info
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CCifXDeviceSys::FindNextFile(uint32_t ulChannel, CIFX_DIRECTORYENTRY* ptDirectoryInfo)
{
  return m_pcCifXAPI->xSysdeviceFindNextFile(m_hSysChannel, ulChannel, ptDirectoryInfo, NULL, NULL);
}

int32_t CCifXDeviceSys::Upload(char* szFilename, uint32_t ulChannel, unsigned char* pbBuffer, uint32_t* pulFileSize)
{
  return m_pcCifXAPI->xSysdeviceUpload(m_hSysChannel, ulChannel, DOWNLOAD_MODE_FILE, szFilename,
                                       pbBuffer, pulFileSize, NULL, NULL, NULL);
}
